/*****************************************************************************
*   LPC54xx SCT PWM Decode for LPCXpresso board
*
*  Use SCT timer capture and capture control features. It implements a
*  PWM decoder which measures the duty cycle of a PWM signal and determines
*  whether it is above (max_width) or below (min_width) a specific value.
*  The PWM signal frequency is assumed to be 10 kHz. Two output signals
*  (width_error and timeout) are included to indicate when the 10 kHz signal
*  has an error or is missing.
*****************************************************************************/
#include "board.h"

/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/
/* Define expected signal characteristics */
#define PWM_FREQUENCY           10000                      // PWM frequency in Hz
#define PWM_RESOLUTION_NS       1000                       // Timer resolution in ns
#define PWM_MIN_DUTY_PERCENT    25                         // Minimum allowed duty cycle in %
#define PWM_MAX_DUTY_PERCENT    70                         // Maximum allowed duty cycle in %

/* Derived constants */
#define SCT_PRESCALER           (((SystemCoreClock / 1000u) * PWM_RESOLUTION_NS) / 1000000u - 1u)

#define match_min_width         ((10000000u * PWM_MIN_DUTY_PERCENT) / (PWM_FREQUENCY * PWM_RESOLUTION_NS))
#define match_max_width         ((10000000u * PWM_MAX_DUTY_PERCENT) / (PWM_FREQUENCY * PWM_RESOLUTION_NS))
#define match_no_input          ((10000000u * 300                 ) / (PWM_FREQUENCY * PWM_RESOLUTION_NS))

/*****************************************************************************
 * Private functions
 ****************************************************************************/
/*****************************************************************************
 * Public functions
 ****************************************************************************/
void SCT_Init(void);


void SCT_Init(void)
{
	Chip_SCT_Init(LPC_SCT);																			// enable the SCT clock

	Chip_SCT_Config(LPC_SCT, SCT_CONFIG_32BIT_COUNTER |
							  SCT_CONFIG_AUTOLIMIT_L 	);											// unified timer and auto limit


	Chip_SCT_SetControl(LPC_SCT, SCT_CTRL_PRE_L(SCT_PRESCALER));									// set pre-scaler

	LPC_SCT->REGMODE_L      = 0x00000018;                 											// 3x MATCH, 2x CAPTURE used

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_0, match_max_width);									// match_max_width
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_0, match_max_width);

	Chip_SCT_SetMatchCount(LPC_SCT,SCT_MATCH_1,match_min_width);									// match_min_width
	Chip_SCT_SetMatchReload(LPC_SCT,SCT_MATCH_0,match_min_width );

	Chip_SCT_SetMatchCount(LPC_SCT,SCT_MATCH_2,match_no_input);										// match_no_input
	Chip_SCT_SetMatchReload(LPC_SCT,SCT_MATCH_2,match_no_input );


	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_0, ENABLE_ALL_STATES);									// event 0 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_0,  (CHIP_SCT_EVENTCTRL_T) ( SCT_EVECTRL_MATCH2 |
                                                                        SCT_IOSEL_1	     |      
                                                                        SCT_COMBMODE_MATCH ));	// related to match_no_input only

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_1, ENABLE_ALL_STATES);									// event 1 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T) (  SCT_IOCOND_RAISE |
                                                                        SCT_IOSEL_1	   |
                                                                        SCT_COMBMODE_IO  )); 		// IN_0 rising edge condition only

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_2, ENABLE_STATE0);									 	// event 2 happens in state 0
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_2, (CHIP_SCT_EVENTCTRL_T) (  SCT_EVECTRL_MATCH1   | 	// related to match_min_width
                                                                        SCT_IOSEL_1	       |
                                                                        SCT_COMBMODE_MATCH   | 	// match condition only
                                                                        SCT_STATELD_1		   | 	// STATEV is loaded into state
                                                                        SCT_STATEEV_1		    ));	// new state is 1

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_3, ENABLE_STATE1);		    							// event 3 happens in state 1
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_3,  (CHIP_SCT_EVENTCTRL_T) ( SCT_IOCOND_FALL  	    | 	// IN_0 falling edge
                                                                        SCT_IOSEL_1	        |
                                                                        SCT_COMBMODE_IO   	|	// IO condition only
                                                                        SCT_STATELD_1		  	| 	// STATEV is loaded into state
                                                                        SCT_STATEEV_0		  	));	// new state is 0

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_4, ENABLE_STATE1);		    							// event 4 happens in state 1
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_4, (CHIP_SCT_EVENTCTRL_T) (  SCT_EVECTRL_MATCH0  | 	    // related to match_max_width
                                                                        SCT_IOSEL_1	      |
                                                                        SCT_COMBMODE_MATCH  |		// match condition only
                                                                        SCT_STATELD_1		  | 	// STATEV is loaded into state
                                                                        SCT_STATEEV_0		 ));	// new state is 0


	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_5, ENABLE_STATE0);		    							// event 5 happens in state 0
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_5, (CHIP_SCT_EVENTCTRL_T) ( SCT_IOCOND_FALL |  		// IN_0 falling edge condition only
                                                                       SCT_IOSEL_1	 |
                                                                       SCT_COMBMODE_IO ));

	Chip_SCT_CapControl(LPC_SCT, SCT_CAPCTRL_3 , SCT_EVT_0);										// event 1 is causing capture 3
	Chip_SCT_CapControl(LPC_SCT, SCT_CAPCTRL_4 , (CHIP_SCT_EVENT_T)  (SCT_EVT_3 |
                                                                      SCT_EVT_5 ));					// event 3 and 5 cause capture 4


	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_5, SCT_EVT_1);											// event 1 set   OUT5 (no timeout)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_5, SCT_EVT_0);											// event 0 clear OUT5 (timeout)

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_4, SCT_EVT_3);											// event 3 set   OUt4 (no width error)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_4, (CHIP_SCT_EVENT_T)  ( SCT_EVT_0 |
                                                                      SCT_EVT_5 ));					// event 0 and 5 clear OUT4 (width error)

	Chip_SCT_Output(LPC_SCT, 0x30);        															// default set OUT4 and OUT5

	LPC_SCT->LIMIT_L        = (1 << 0) | (1 << 1);        											// event 0 and 1 limit the timer

	Chip_SCT_EnableEventInt(LPC_SCT, (CHIP_SCT_EVENT_T)	( SCT_EVT_0 |
                                                          SCT_EVT_5 ));	   							// event 0 and 5 generate an irq


	NVIC_EnableIRQ(SCT0_IRQn);                             											// enable SCT interrupt

	Chip_SCT_ClearControl(LPC_SCT,SCT_CTRL_HALT_L);	   												// unhalt it by clearing bit 2 of the CTRL register
}



void SCT0_IRQHandler(void)
{
  uint32_t status = Chip_SCT_GetEventFlag(LPC_SCT);

    if (status & 1)                                    												// EVENT 0: timeout
    {
		/* New measurement result */
		;
    }

    if (status & (1 << 5))                             												// EVENT 5: width error
    {
		/* Time-out (no signal) */
		;
    }
    Chip_SCT_SetEventFlag(LPC_SCT, (CHIP_SCT_EVENT_T)(status));		   								// Acknowledge interrupts
}


int main(void)
{
	volatile static int i = 0;                           											// force counter to be placed into memory

	SystemCoreClockUpdate();
	Board_Init();


	Chip_Clock_EnablePeriphClock(	(CHIP_SYSCON_CLOCK_T)   ( SYSCON_CLOCK_GPIO0 	|				// enable GPIO port 0 clock
                                                          	  SYSCON_CLOCK_GPIO1	| 				// enable GPIO port 1 clock
                                                          	  SYSCON_CLOCK_INPUTMUX 	  	|				// enable PMUX clock
                                                          	  SYSCON_CLOCK_IOCON	));				//enable IOCON clock


	Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 2, IOCON_FUNC3 | IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);		//SCT0_OUT5 = P1.2  = timeout(green   LED)
	Chip_IOCON_PinMuxSet(LPC_IOCON, 1, 1, IOCON_FUNC3 | IOCON_MODE_INACT | IOCON_DIGITAL_EN | IOCON_INPFILT_OFF);		//SCT0_OUT4 = P1.1  = width_error(blue   LED)

	Chip_GPIO_SetPinDIRInput(LPC_GPIO, 0, 24);


	Chip_IOCON_PinMuxSet(LPC_IOCON, 0, 24, 	IOCON_FUNC0 		  |
                                          IOCON_I2C_SLEW		|
                                          IOCON_DIGITAL_EN	|
                                          IOCON_INPFILT_OFF	|
                                          0x0 << 6			    |
                                          0x0 << 9			    |
                                          0x1 << 10			    );									// SCT0_IN1  = P0.24 = SW1

    SCT_Init();                                          											// Initialize SCT0

    while (1)                                              											// loop forever
    {
    	/* The settings are chosen in % of duty cycle (assuming a PWM frequency of 10 kHz).
    	* (In a real world application you want to select better resolution!)
    	*/
    	        i = LPC_SCT->CAP[4].U;

    	        __WFI();
    }
}
